/**
 * Google SMS 验证提取器 - 内容脚本
 * 显示提取结果的浮动面板
 */

console.log('[SMS Extractor] 内容脚本已加载');
console.log('[SMS Extractor] 当前 URL:', location.href);

// 只在 devicephoneverification 页面显示等待面板
// mophoneverification 是二维码页面，由 qrcode-scanner.js 处理
const isSmsVerificationPage = location.href.includes('devicephoneverification');

if (isSmsVerificationPage) {
  console.log('[SMS Extractor] ✅ 检测到 SMS 验证页面');
  
  // 延迟显示等待面板
  setTimeout(() => {
    showWaitingPanel();
  }, 500);
} else {
  console.log('[SMS Extractor] 不是 SMS 验证页面，跳过');
}

// 显示等待面板
function showWaitingPanel() {
  const existing = document.getElementById('sms-extractor-panel');
  if (existing) return;

  // 创建样式
  if (!document.getElementById('sms-extractor-styles')) {
    const style = document.createElement('style');
    style.id = 'sms-extractor-styles';
    style.textContent = `
      #sms-extractor-panel {
        position: fixed;
        top: 20px;
        right: 20px;
        width: 400px;
        max-width: calc(100vw - 40px);
        background: white;
        border: 2px solid #667eea;
        border-radius: 12px;
        box-shadow: 0 8px 24px rgba(0,0,0,0.15);
        z-index: 2147483647;
        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Arial, sans-serif;
        animation: slideIn 0.3s ease-out;
      }
      @keyframes slideIn {
        from { transform: translateX(100%); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
      }
      @keyframes pulse {
        0%, 100% { opacity: 1; }
        50% { opacity: 0.5; }
      }
      .sms-header {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        padding: 16px 20px;
        border-radius: 10px 10px 0 0;
        display: flex;
        justify-content: space-between;
        align-items: center;
      }
      .sms-header h3 {
        margin: 0;
        font-size: 16px;
        font-weight: 600;
      }
      .sms-close {
        background: rgba(255,255,255,0.2);
        border: none;
        color: white;
        font-size: 20px;
        cursor: pointer;
        width: 28px;
        height: 28px;
        border-radius: 50%;
        transition: background 0.2s;
      }
      .sms-close:hover {
        background: rgba(255,255,255,0.3);
      }
      .sms-body {
        padding: 20px;
      }
      .sms-waiting {
        text-align: center;
        padding: 30px 20px;
      }
      .sms-waiting-icon {
        font-size: 48px;
        margin-bottom: 16px;
        animation: pulse 2s infinite;
      }
      .sms-waiting-text {
        color: #666;
        font-size: 14px;
        margin: 0;
      }
      .sms-result {
        display: none;
      }
      .sms-field {
        margin-bottom: 16px;
      }
      .sms-field:last-child {
        margin-bottom: 0;
      }
      .sms-label {
        font-size: 12px;
        color: #666;
        font-weight: 500;
        margin-bottom: 6px;
        display: block;
      }
      .sms-value {
        background: #f7f7f7;
        padding: 12px;
        border-radius: 6px;
        font-family: 'Monaco', 'Menlo', monospace;
        font-size: 14px;
        color: #333;
        word-break: break-all;
        border: 1px solid #e0e0e0;
      }
      .sms-token {
        background: #fff9e6;
        border: 1px solid #ffd700;
      }
      .sms-token .sms-value {
        background: #fff9e6;
        color: #d97706;
        font-size: 18px;
        font-weight: 600;
        text-align: center;
        letter-spacing: 1px;
      }
      .sms-copy-btn {
        background: #667eea;
        color: white;
        border: none;
        padding: 8px 16px;
        border-radius: 6px;
        font-size: 13px;
        cursor: pointer;
        margin-top: 8px;
        transition: background 0.2s;
      }
      .sms-copy-btn:hover {
        background: #5568d3;
      }
      .sms-copy-btn:active {
        background: #4450b5;
      }
    `;
    document.head.appendChild(style);
  }

  // 创建面板
  const panel = document.createElement('div');
  panel.id = 'sms-extractor-panel';
  panel.innerHTML = `
    <div class="sms-header">
      <h3>📱 SMS 验证提取器</h3>
      <button class="sms-close">×</button>
    </div>
    <div class="sms-body">
      <div class="sms-waiting">
        <div class="sms-waiting-icon">⏳</div>
        <p class="sms-waiting-text">等待点击 "Send SMS" 按钮...</p>
      </div>
      <div class="sms-result">
        <div class="sms-field">
          <span class="sms-label">目标号码</span>
          <div class="sms-value" id="sms-phone"></div>
        </div>
        <div class="sms-field">
          <span class="sms-label">短信内容</span>
          <div class="sms-value" id="sms-body"></div>
        </div>
        <div class="sms-field sms-token">
          <span class="sms-label">🔑 验证令牌</span>
          <div class="sms-value" id="sms-token"></div>
          <button class="sms-copy-btn" id="sms-copy">📋 复制令牌</button>
        </div>
      </div>
    </div>
  `;
  
  document.body.appendChild(panel);
  
  // 关闭按钮
  panel.querySelector('.sms-close').addEventListener('click', () => {
    panel.remove();
  });
}

// 显示提取结果
function displayResult(data) {
  console.log('[SMS Extractor] 显示提取结果:', data);
  
  // 保存到 chrome.storage 供 popup 使用
  chrome.storage.local.set({
    lastResult: data,
    timestamp: Date.now()
  }, () => {
    console.log('[SMS Extractor] 数据已保存到 storage');
  });
  
  const panel = document.getElementById('sms-extractor-panel');
  if (!panel) return;
  
  const waiting = panel.querySelector('.sms-waiting');
  const result = panel.querySelector('.sms-result');
  
  // 隐藏等待，显示结果
  waiting.style.display = 'none';
  result.style.display = 'block';
  
  // 填充数据
  document.getElementById('sms-phone').textContent = data.phoneNumber;
  document.getElementById('sms-body').textContent = data.smsBody;
  document.getElementById('sms-token').textContent = data.verificationToken;
  
  // 复制按钮
  document.getElementById('sms-copy').addEventListener('click', () => {
    navigator.clipboard.writeText(data.verificationToken).then(() => {
      const btn = document.getElementById('sms-copy');
      const originalText = btn.textContent;
      btn.textContent = '✅ 已复制';
      setTimeout(() => {
        btn.textContent = originalText;
      }, 2000);
    });
  });
}

// 处理 SMS URI
function handleSmsUri(uri) {
  console.log('[SMS Extractor] 处理 SMS URI:', uri);
  
  // 使用 parser.js 中的解析函数
  if (typeof window.SmsParser !== 'undefined' && typeof window.SmsParser.parse === 'function') {
    console.log('[SMS Extractor] 调用 SmsParser.parse');
    const result = window.SmsParser.parse(uri);
    console.log('[SMS Extractor] 解析结果:', result);
    if (result.success) {
      displayResult(result);
    } else {
      console.error('[SMS Extractor] 解析失败:', result.error);
    }
  } else {
    console.error('[SMS Extractor] SmsParser 未加载');
  }
}

// 监听来自 super-early-inject.js 的消息
window.addEventListener('message', (event) => {
  if (event.source === window && event.data && event.data.type === 'SMS_URI_CAPTURED') {
    console.log('[SMS Extractor] 收到拦截器消息:', event.data.url);
    handleSmsUri(event.data.url);
  }
});

console.log('[SMS Extractor] 监听已启动');
