/**
 * Google SMS 验证提取器 - SMS URI 解析器
 * @version 1.0.0
 * @description 解析 Google 设备验证的 SMS URI，提取电话号码、短信内容和验证令牌
 */

(function(global) {
  'use strict';

  /**
   * SMS URI 解析器
   * @param {string} smsUri - 完整的 SMS URI
   * @returns {Object} 解析结果
   */
  function parseSmsUri(smsUri) {
    const result = {
      success: false,
      phoneNumber: null,
      smsBody: null,
      verificationToken: null,
      rawUri: smsUri,
      error: null
    };

    try {
      // 1. 输入验证
      if (!smsUri || typeof smsUri !== 'string') {
        throw new Error('无效的 URI 格式');
      }

      // 2. 长度限制（防止恶意输入）
      if (smsUri.length > 2000) {
        throw new Error('URI 过长');
      }

      // 3. 格式验证
      if (!smsUri.startsWith('sms://') && !smsUri.startsWith('sms:')) {
        throw new Error('不是有效的 SMS URI');
      }

      // 4. 安全检查（防止 XSS）
      const dangerousPatterns = /<script|javascript:|onerror|onclick|onload/i;
      if (dangerousPatterns.test(smsUri)) {
        throw new Error('检测到危险内容');
      }

      // 5. 提取电话号码
      // 支持多种格式: sms://12345/ 或 sms://12345? 或 sms://12345&
      const phonePatterns = [
        /sms:\/\/(\d+)\//,
        /sms:\/\/(\d+)\?/,
        /sms:\/\/(\d+)&/,
        /sms:\/\/(\d+)$/,
        /sms:(\d+)\?/,
        /sms:(\d+)&/
      ];

      let phoneNumber = null;
      for (const pattern of phonePatterns) {
        const match = smsUri.match(pattern);
        if (match) {
          phoneNumber = match[1];
          break;
        }
      }

      if (!phoneNumber) {
        throw new Error('无法提取电话号码');
      }

      // 验证电话号码格式（通常是 5 位短代码）
      if (!/^\d{4,10}$/.test(phoneNumber)) {
        throw new Error('电话号码格式无效');
      }

      // 6. 提取消息体
      // 支持 body= 或 text= 参数
      const bodyPatterns = [
        /[?&]body=([^&]+)/,
        /[?&]text=([^&]+)/
      ];

      let encodedBody = null;
      for (const pattern of bodyPatterns) {
        const match = smsUri.match(pattern);
        if (match) {
          encodedBody = match[1];
          break;
        }
      }

      if (!encodedBody) {
        throw new Error('无法提取短信内容');
      }

      // 7. URL 解码
      let smsBody;
      try {
        smsBody = decodeURIComponent(encodedBody);
      } catch (e) {
        // 如果 decodeURIComponent 失败，尝试手动解码
        smsBody = encodedBody.replace(/\+/g, ' ').replace(/%([0-9A-F]{2})/gi, (match, hex) => {
          return String.fromCharCode(parseInt(hex, 16));
        });
      }

      // 8. 提取验证令牌（括号内的内容）
      const tokenMatch = smsBody.match(/\(([^)]+)\)/);
      const verificationToken = tokenMatch ? tokenMatch[1] : null;

      // 9. 验证令牌格式（如果存在）
      if (verificationToken) {
        // 通常是 8-16 位字母数字组合
        if (verificationToken.length < 6 || verificationToken.length > 20) {
          console.warn('验证令牌长度异常:', verificationToken.length);
        }
      }

      // 10. 返回成功结果
      result.success = true;
      result.phoneNumber = phoneNumber;
      result.smsBody = smsBody;
      result.verificationToken = verificationToken;

    } catch (error) {
      result.error = error.message;
      console.error('[SMS Parser Error]', error);
    }

    return result;
  }

  /**
   * 验证解析结果的完整性
   * @param {Object} result - 解析结果
   * @returns {boolean} 是否有效
   */
  function validateResult(result) {
    if (!result.success) {
      return false;
    }

    if (!result.phoneNumber || !result.smsBody) {
      return false;
    }

    return true;
  }

  /**
   * 格式化解析结果为文本
   * @param {Object} result - 解析结果
   * @returns {string} 格式化的文本
   */
  function formatResult(result) {
    if (!result.success) {
      return `解析失败: ${result.error}`;
    }

    return `目标号码: ${result.phoneNumber}\n短信内容: ${result.smsBody}\n验证令牌: ${result.verificationToken || 'N/A'}`;
  }

  /**
   * 将解析结果转换为 JSON
   * @param {Object} result - 解析结果
   * @param {boolean} pretty - 是否美化输出
   * @returns {string} JSON 字符串
   */
  function resultToJson(result, pretty = false) {
    const data = {
      success: result.success,
      phoneNumber: result.phoneNumber,
      smsBody: result.smsBody,
      verificationToken: result.verificationToken,
      timestamp: new Date().toISOString()
    };

    if (!result.success) {
      data.error = result.error;
    }

    return JSON.stringify(data, null, pretty ? 2 : 0);
  }

  // 导出到全局对象
  const SmsParser = {
    parse: parseSmsUri,
    validate: validateResult,
    format: formatResult,
    toJson: resultToJson,
    version: '1.0.0'
  };

  // 根据环境选择导出方式
  if (typeof module !== 'undefined' && module.exports) {
    // Node.js / CommonJS
    module.exports = SmsParser;
  } else if (typeof define === 'function' && define.amd) {
    // AMD
    define(function() { return SmsParser; });
  } else {
    // 浏览器全局对象
    global.SmsParser = SmsParser;
  }

})(typeof window !== 'undefined' ? window : this);

