/**
 * QR Code Scanner - 自动识别页面二维码并跳转
 * 策略：
 * 1. 优先使用 Barcode Detection API（如果浏览器支持）
 * 2. 否则尝试从 DOM 中查找已解析的 URL（Google 可能在某些属性中存储）
 * 3. 提供手动输入选项
 */

console.log('='.repeat(60));
console.log('[QR Scanner] 📱📱📱 二维码扫描器文件已加载');
console.log('[QR Scanner] 版本: 1.1.0');
console.log('[QR Scanner] 加载时间:', new Date().toISOString());
console.log('[QR Scanner] 当前 URL:', window.location.href);
console.log('='.repeat(60));

// 检查是否是二维码页面
function isQRCodePage() {
  const url = window.location.href;
  
  console.log('[QR Scanner] 检查 URL:', url);
  
  // 明确排除：只有 devicephoneverification 才是 SMS 验证页面
  // mophoneverification 是显示二维码的页面
  if (url.includes('/devicephoneverification/')) {
    console.log('[QR Scanner] 这是 devicephoneverification（SMS 验证页面），跳过');
    return false;
  }
  
  const bodyText = document.body.innerText || document.body.textContent || '';
  console.log('[QR Scanner] 页面文本片段:', bodyText.substring(0, 200));
  
  const hasQRText = (
    bodyText.includes('扫描二维码') ||
    bodyText.includes('Scan the QR code') ||
    bodyText.includes('使用您的手机扫描二维码') ||
    bodyText.includes('需要先验证') ||
    bodyText.includes('Verify some info') ||
    bodyText.includes('然后才能创建账号')
  );
  
  // 检查是否有二维码图片
  const hasQRImage = (
    document.querySelector('img[alt*="QR"]') !== null ||
    document.querySelector('img[alt*="code"]') !== null ||
    document.querySelector('canvas') !== null ||
    Array.from(document.querySelectorAll('img')).some(img => {
      return img.width > 200 && img.height > 200 && Math.abs(img.width - img.height) < 50;
    })
  );
  
  // 检查 URL 模式：mophoneverification 通常是二维码页面
  const urlIndicatesQR = (
    url.includes('/mophoneverification/') ||
    url.includes('/signup/') && url.includes('phoneverification')
  );
  
  console.log('[QR Scanner] 检测结果:', {
    hasQRText,
    hasQRImage,
    urlIndicatesQR
  });
  
  return hasQRText || hasQRImage || urlIndicatesQR;
}

// 从 DOM 中查找可能的 URL（Google 可能存储在某些地方）
function findURLInDOM() {
  console.log('[QR Scanner] 在 DOM 中搜索 URL...');
  
  // 检查所有可能包含 URL 的属性
  const elements = document.querySelectorAll('[data-url], [data-href], [data-link]');
  for (const el of elements) {
    const url = el.dataset.url || el.dataset.href || el.dataset.link;
    if (url && url.includes('devicephoneverification')) {
      console.log('[QR Scanner] 在 data 属性中找到 URL:', url);
      return url;
    }
  }
  
  // 检查所有链接
  const links = document.querySelectorAll('a[href]');
  for (const link of links) {
    if (link.href.includes('devicephoneverification')) {
      console.log('[QR Scanner] 在链接中找到 URL:', link.href);
      return link.href;
    }
  }
  
  // 检查页面中的所有文本节点（URL 可能以文本形式存在）
  const walker = document.createTreeWalker(
    document.body,
    NodeFilter.SHOW_TEXT,
    null,
    false
  );
  
  let node;
  while (node = walker.nextNode()) {
    const text = node.textContent;
    const urlMatch = text.match(/https:\/\/accounts\.google\.com\/devicephoneverification\/[^\s"']+/);
    if (urlMatch) {
      console.log('[QR Scanner] 在文本中找到 URL:', urlMatch[0]);
      return urlMatch[0];
    }
  }
  
  console.log('[QR Scanner] 在 DOM 中未找到 URL');
  return null;
}

// 查找二维码图片
function findQRCodeImages() {
  const images = [];
  
  // 查找所有可能的二维码图片
  document.querySelectorAll('img, canvas').forEach(el => {
    const width = el.width || el.naturalWidth || 0;
    const height = el.height || el.naturalHeight || 0;
    
    // 二维码通常是正方形，尺寸 > 200px
    if (width > 200 && height > 200 && Math.abs(width - height) < 50) {
      images.push(el);
      console.log('[QR Scanner] 找到候选图片:', el, `${width}x${height}`);
    }
  });
  
  return images;
}

// 使用 Barcode Detection API 扫描
async function scanWithBarcodeAPI(element) {
  if (!('BarcodeDetector' in window)) {
    return null;
  }
  
  try {
    console.log('[QR Scanner] 使用 Barcode Detection API 扫描...');
    const barcodeDetector = new BarcodeDetector({ formats: ['qr_code'] });
    const barcodes = await barcodeDetector.detect(element);
    
    if (barcodes.length > 0) {
      console.log('[QR Scanner] ✅ 识别成功:', barcodes[0].rawValue);
      return barcodes[0].rawValue;
    }
  } catch (e) {
    console.error('[QR Scanner] Barcode Detection 失败:', e);
  }
  
  return null;
}

// 主扫描函数
async function scanPageForQRCode() {
  console.log('[QR Scanner] 开始扫描...');
  
  // 方法 1: 在 DOM 中查找 URL
  const domUrl = findURLInDOM();
  if (domUrl) {
    return domUrl;
  }
  
  // 方法 2: 使用 Barcode Detection API
  const images = findQRCodeImages();
  for (const img of images) {
    const url = await scanWithBarcodeAPI(img);
    if (url) {
      return url;
    }
  }
  
  return null;
}

// 显示扫描面板
function showScanPanel() {
  const panel = document.createElement('div');
  panel.id = 'qr-scan-panel';
  panel.style.cssText = `
    position: fixed;
    top: 20px;
    right: 20px;
    width: 350px;
    background: white;
    border: 2px solid #667eea;
    border-radius: 12px;
    box-shadow: 0 8px 24px rgba(0,0,0,0.2);
    z-index: 2147483647;
    padding: 20px;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Arial, sans-serif;
  `;
  
  panel.innerHTML = `
    <div style="text-align: center;">
      <div style="font-size: 32px; margin-bottom: 12px;">📱</div>
      <h3 style="margin: 0 0 12px 0; font-size: 18px; color: #333;">
        二维码扫描器
      </h3>
      <div id="scan-status" style="color: #666; font-size: 14px; margin-bottom: 16px;">
        正在扫描二维码...
      </div>
      <div style="margin-bottom: 16px;">
        <button id="manual-scan-btn" style="
          width: 100%;
          padding: 10px;
          background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
          color: white;
          border: none;
          border-radius: 8px;
          font-size: 14px;
          font-weight: 600;
          cursor: pointer;
          margin-bottom: 8px;
        ">
          🔄 重新扫描
        </button>
        <button id="manual-input-btn" style="
          width: 100%;
          padding: 10px;
          background: #f0f0f0;
          color: #666;
          border: none;
          border-radius: 8px;
          font-size: 14px;
          cursor: pointer;
        ">
          ✍️ 手动输入 URL
        </button>
      </div>
      <button id="close-panel-btn" style="
        width: 100%;
        padding: 8px;
        background: transparent;
        color: #999;
        border: none;
        font-size: 12px;
        cursor: pointer;
      ">
        关闭
      </button>
    </div>
  `;
  
  document.body.appendChild(panel);
  
  // 按钮事件
  document.getElementById('manual-scan-btn').addEventListener('click', async () => {
    document.getElementById('scan-status').textContent = '正在重新扫描...';
    const url = await scanPageForQRCode();
    if (url) {
      showRedirectPrompt(url);
      panel.remove();
    } else {
      document.getElementById('scan-status').textContent = '❌ 未识别到二维码，请手动输入';
    }
  });
  
  document.getElementById('manual-input-btn').addEventListener('click', () => {
    showManualInputDialog();
    panel.remove();
  });
  
  document.getElementById('close-panel-btn').addEventListener('click', () => {
    panel.remove();
  });
  
  return panel;
}

// 显示手动输入对话框
function showManualInputDialog() {
  const dialog = document.createElement('div');
  dialog.id = 'qr-manual-input';
  dialog.style.cssText = `
    position: fixed;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    width: 500px;
    max-width: 90%;
    background: white;
    border: 2px solid #667eea;
    border-radius: 12px;
    box-shadow: 0 8px 24px rgba(0,0,0,0.3);
    z-index: 2147483647;
    padding: 24px;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Arial, sans-serif;
  `;
  
  dialog.innerHTML = `
    <h3 style="margin: 0 0 16px 0; font-size: 20px; color: #333;">
      手动输入验证页面 URL
    </h3>
    <p style="color: #666; font-size: 14px; margin: 0 0 16px 0;">
      请在手机上扫描二维码，然后将获取的 URL 粘贴到下方：
    </p>
    <input type="text" id="manual-url-input" placeholder="https://accounts.google.com/devicephoneverification/..." style="
      width: 100%;
      padding: 12px;
      border: 2px solid #ddd;
      border-radius: 8px;
      font-size: 14px;
      margin-bottom: 16px;
      box-sizing: border-box;
    ">
    <div style="display: flex; gap: 12px;">
      <button id="confirm-url-btn" style="
        flex: 1;
        padding: 12px;
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        border: none;
        border-radius: 8px;
        font-size: 16px;
        font-weight: 600;
        cursor: pointer;
      ">
        确认跳转
      </button>
      <button id="cancel-input-btn" style="
        flex: 1;
        padding: 12px;
        background: #f0f0f0;
        color: #666;
        border: none;
        border-radius: 8px;
        font-size: 16px;
        cursor: pointer;
      ">
        取消
      </button>
    </div>
  `;
  
  document.body.appendChild(dialog);
  
  const input = document.getElementById('manual-url-input');
  input.focus();
  
  document.getElementById('confirm-url-btn').addEventListener('click', () => {
    const url = input.value.trim();
    if (url && url.startsWith('http')) {
      window.location.href = url;
    } else {
      alert('请输入有效的 URL');
    }
  });
  
  document.getElementById('cancel-input-btn').addEventListener('click', () => {
    dialog.remove();
  });
  
  // 回车提交
  input.addEventListener('keypress', (e) => {
    if (e.key === 'Enter') {
      document.getElementById('confirm-url-btn').click();
    }
  });
}

// 显示跳转提示
function showRedirectPrompt(url) {
  const panel = document.createElement('div');
  panel.id = 'qr-redirect-panel';
  panel.style.cssText = `
    position: fixed;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    width: 450px;
    max-width: 90%;
    background: white;
    border: 2px solid #667eea;
    border-radius: 12px;
    box-shadow: 0 8px 24px rgba(0,0,0,0.3);
    z-index: 2147483647;
    padding: 24px;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Arial, sans-serif;
  `;
  
  panel.innerHTML = `
    <div style="text-align: center;">
      <div style="font-size: 48px; margin-bottom: 16px;">✅</div>
      <h3 style="margin: 0 0 16px 0; font-size: 20px; color: #333;">
        检测到验证页面 URL
      </h3>
      <div style="
        background: #f5f5f5;
        padding: 12px;
        border-radius: 8px;
        margin-bottom: 20px;
        font-size: 12px;
        color: #666;
        word-break: break-all;
        max-height: 100px;
        overflow-y: auto;
      ">
        ${url}
      </div>
      <button id="qr-redirect-btn" style="
        width: 100%;
        padding: 14px;
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        border: none;
        border-radius: 8px;
        font-size: 16px;
        font-weight: 600;
        cursor: pointer;
        margin-bottom: 8px;
      ">
        🚀 立即跳转 (3s)
      </button>
      <button id="qr-cancel-btn" style="
        width: 100%;
        padding: 12px;
        background: #f0f0f0;
        color: #666;
        border: none;
        border-radius: 8px;
        font-size: 14px;
        cursor: pointer;
      ">
        取消
      </button>
    </div>
  `;
  
  document.body.appendChild(panel);
  
  // 跳转按钮
  document.getElementById('qr-redirect-btn').addEventListener('click', () => {
    console.log('[QR Scanner] 用户确认跳转');
    window.location.href = url;
  });
  
  // 取消按钮
  document.getElementById('qr-cancel-btn').addEventListener('click', () => {
    console.log('[QR Scanner] 用户取消跳转');
    panel.remove();
  });
  
  // 3 秒后自动跳转
  let countdown = 3;
  const countdownInterval = setInterval(() => {
    countdown--;
    const btn = document.getElementById('qr-redirect-btn');
    if (btn) {
      btn.textContent = countdown > 0 ? `🚀 立即跳转 (${countdown}s)` : '🚀 正在跳转...';
    }
    if (countdown <= 0) {
      clearInterval(countdownInterval);
      console.log('[QR Scanner] 自动跳转');
      window.location.href = url;
    }
  }, 1000);
}

// 初始化
async function init() {
  console.log('🚀🚀🚀 [QR Scanner] 初始化开始');
  console.log('[QR Scanner] 当前 URL:', window.location.href);
  console.log('[QR Scanner] document.readyState:', document.readyState);
  
  // 等待页面加载
  if (document.readyState !== 'complete') {
    console.log('[QR Scanner] 等待页面加载完成...');
    await new Promise(resolve => window.addEventListener('load', resolve));
    console.log('[QR Scanner] 页面加载完成');
  }
  
  // 检查是否是二维码页面
  const isQR = isQRCodePage();
  console.log('[QR Scanner] isQRCodePage() 返回:', isQR);
  
  if (!isQR) {
    console.log('[QR Scanner] ❌ 不是二维码页面，退出');
    return;
  }
  
  console.log('[QR Scanner] ✅✅✅ 确认是二维码页面');
  
  // 显示扫描面板
  console.log('[QR Scanner] 显示扫描面板...');
  const panel = showScanPanel();
  console.log('[QR Scanner] 扫描面板已显示');
  
  // 等待 1 秒确保二维码渲染完成
  console.log('[QR Scanner] 等待 1 秒...');
  await new Promise(resolve => setTimeout(resolve, 1000));
  
  // 尝试自动扫描
  console.log('[QR Scanner] 开始自动扫描...');
  const url = await scanPageForQRCode();
  console.log('[QR Scanner] 扫描结果:', url);
  
  if (url && url.startsWith('http')) {
    console.log('[QR Scanner] ✅✅✅ 自动扫描成功:', url);
    console.log('[QR Scanner] 移除扫描面板');
    panel.remove();
    console.log('[QR Scanner] 显示跳转提示');
    showRedirectPrompt(url);
  } else {
    console.log('[QR Scanner] ❌ 自动扫描失败，等待用户操作');
    const statusEl = document.getElementById('scan-status');
    if (statusEl) {
      statusEl.textContent = '❌ 未能自动识别，请重试或手动输入';
    }
  }
  
  console.log('[QR Scanner] 初始化完成');
}

// 启动（添加错误捕获）
async function safeInit() {
  try {
    console.log('[QR Scanner] 🚀 开始启动...');
    await init();
  } catch (error) {
    console.error('[QR Scanner] ❌ 初始化失败:', error);
    console.error('[QR Scanner] 错误堆栈:', error.stack);
  }
}

// 监听 URL 变化（SPA 导航）
let lastUrl = window.location.href;
const urlObserver = new MutationObserver(() => {
  const currentUrl = window.location.href;
  if (currentUrl !== lastUrl) {
    console.log('[QR Scanner] 🔄 URL 变化:', lastUrl, '→', currentUrl);
    lastUrl = currentUrl;
    // URL 变化后重新初始化
    setTimeout(safeInit, 500);
  }
});

// 观察 URL 变化
urlObserver.observe(document, { subtree: true, childList: true });

// 初次加载
if (document.readyState === 'loading') {
  console.log('[QR Scanner] 等待 DOMContentLoaded...');
  document.addEventListener('DOMContentLoaded', safeInit);
} else {
  console.log('[QR Scanner] 页面已加载，直接启动');
  safeInit();
}

// 也监听 popstate 事件（后退/前进按钮）
window.addEventListener('popstate', () => {
  console.log('[QR Scanner] 🔙 检测到 popstate 事件');
  setTimeout(safeInit, 500);
});
