/**
 * Google SMS 验证提取器 - 核心拦截脚本
 * 在页面 JS 执行前运行，拦截 Google API 响应并提取 SMS 信息
 */

(function() {
  'use strict';
  
  console.log('🚀🚀🚀 [SMS Extractor] 拦截器已注入 - ' + new Date().toISOString());
  console.log('[SMS Extractor] 当前 URL:', window.location.href);
  
  // 存储捕获的 SMS URI
  let capturedSmsUri = null;
  
  // 从 API 响应中提取 SMS 信息
  function extractSmsFromApiResponse(text) {
    if (!text || typeof text !== 'string') return null;
    
    try {
      // 匹配 Google API 返回的格式: ["96831","Send this message..."]
      const apiMatch = text.match(/\[\\?"(\d{5,6})\\?",\\?"([^"]+)\\?"\]/);
      if (apiMatch) {
        console.log('✅ [SMS Extractor] 匹配到 API 数据:', apiMatch);
        const phoneNumber = apiMatch[1];
        let smsBody = apiMatch[2];
        
        // 清理转义字符
        smsBody = smsBody
          .replace(/\\"/g, '"')
          .replace(/\\'/g, "'")
          .replace(/\\\\/g, '\\')
          .replace(/\\n/g, '\n')
          .replace(/\\r/g, '\r')
          .replace(/\\t/g, '\t')
          .replace(/\\+$/, ''); // 移除末尾多余的反斜杠
        
        // 构造 sms:// URI
        const encodedBody = encodeURIComponent(smsBody);
        const uri = `sms://${phoneNumber}/&body=${encodedBody}`;
        
        console.log('✅✅✅ [SMS Extractor] 从 API 响应提取到 SMS URI:', uri);
        return uri;
      }
    } catch (e) {
      console.error('❌ [SMS Extractor] 解析失败:', e);
    }
    
    return null;
  }
  
  // 处理捕获的 SMS URI
  function handleCapturedUri(uri) {
    if (!uri) {
      console.log('⚠️ [SMS Extractor] URI 为空，跳过');
      return;
    }
    
    if (capturedSmsUri === uri) {
      console.log('⚠️ [SMS Extractor] URI 已捕获过，跳过');
      return;
    }
    
    console.log('🎯🎯🎯 [SMS Extractor] 捕获新的 SMS URI:', uri);
    capturedSmsUri = uri;
    
    // 发送消息给 content script
    console.log('[SMS Extractor] 发送 postMessage 给 content script');
    window.postMessage({ 
      type: 'SMS_URI_CAPTURED', 
      url: uri
    }, '*');
    console.log('[SMS Extractor] postMessage 已发送');
  }
  
  // 拦截 fetch API
  console.log('[SMS Extractor] 设置 fetch 拦截器');
  const originalFetch = window.fetch;
  window.fetch = function(...args) {
    const url = args[0];
    console.log('[SMS Extractor] fetch 调用:', url);
    
    return originalFetch.apply(this, args).then(response => {
      // 克隆响应以便读取
      const clonedResponse = response.clone();
      
      // 检查是否是 batchexecute API
      if (typeof url === 'string' && url.includes('batchexecute')) {
        console.log('🔍 [SMS Extractor] 检测到 batchexecute API 调用:', url);
        clonedResponse.text().then(text => {
          console.log('[SMS Extractor] 获取响应文本，长度:', text.length);
          console.log('[SMS Extractor] 响应前 500 字符:', text.substring(0, 500));
          const uri = extractSmsFromApiResponse(text);
          if (uri) {
            handleCapturedUri(uri);
          } else {
            console.log('⚠️ [SMS Extractor] 未从响应中提取到 SMS URI');
          }
        }).catch(e => {
          console.error('❌ [SMS Extractor] 读取响应失败:', e);
        });
      }
      
      return response;
    });
  };
  console.log('✅ [SMS Extractor] fetch 拦截器已设置');
  
  // 拦截 XMLHttpRequest
  console.log('[SMS Extractor] 设置 XHR 拦截器');
  const originalXHROpen = XMLHttpRequest.prototype.open;
  const originalXHRSend = XMLHttpRequest.prototype.send;
  
  XMLHttpRequest.prototype.open = function(method, url, ...args) {
    this._url = url;
    console.log('[SMS Extractor] XHR open:', method, url);
    return originalXHROpen.apply(this, [method, url, ...args]);
  };
  
  XMLHttpRequest.prototype.send = function(...args) {
    const xhr = this;
    
    xhr.addEventListener('load', function() {
      if (xhr._url && xhr._url.includes('batchexecute')) {
        console.log('🔍 [SMS Extractor] XHR batchexecute 响应:', xhr._url);
        console.log('[SMS Extractor] 响应状态:', xhr.status);
        console.log('[SMS Extractor] 响应文本长度:', xhr.responseText.length);
        console.log('[SMS Extractor] 响应前 500 字符:', xhr.responseText.substring(0, 500));
        const uri = extractSmsFromApiResponse(xhr.responseText);
        if (uri) {
          handleCapturedUri(uri);
        } else {
          console.log('⚠️ [SMS Extractor] 未从 XHR 响应中提取到 SMS URI');
        }
      }
    });
    
    return originalXHRSend.apply(this, args);
  };
  console.log('✅ [SMS Extractor] XHR 拦截器已设置');
  
  // 暴露全局方法供调试使用
  window.__getSmsUri = function() {
    console.log('[SMS Extractor] 当前捕获的 URI:', capturedSmsUri);
    return capturedSmsUri;
  };
  
  window.__testPostMessage = function() {
    console.log('[SMS Extractor] 测试 postMessage');
    window.postMessage({ 
      type: 'SMS_URI_CAPTURED', 
      url: 'sms://12345/&body=test'
    }, '*');
    console.log('[SMS Extractor] 测试消息已发送');
  };
  
  console.log('✅✅✅ [SMS Extractor] 拦截器已完全就绪，监听 batchexecute API');
  console.log('[SMS Extractor] 调试命令: window.__getSmsUri() 查看捕获的 URI');
  console.log('[SMS Extractor] 调试命令: window.__testPostMessage() 测试消息传递');
})();
